import { supabase } from '@/config/supabase';
import {
  Customer,
  VendorCustomer,
  Measurement,
  AddOrLinkCustomerRequest,
  UpdateCustomerRequest,
  MeasurementRequest
} from './types';

export const customerService = {
  // List customers for vendor, with search/tag
  async listCustomers(vendor_id: string, search?: string, tag?: string) {
    let query = supabase
      .from('vendor_customers')
      .select('*,customer:customer_id(*),measurements:customer_id(id,vendor_id,customer_id,bust,waist,hip,created_at,updated_at)')
      .eq('vendor_id', vendor_id);
    if (tag) query = query.contains('tags', [tag]);
    const { data, error } = await query;
    if (error) return { error };
    let filtered = data;
    if (search) {
      const s = search.toLowerCase();
      filtered = data.filter((vc: any) =>
        vc.customer.full_name.toLowerCase().includes(s) ||
        vc.customer.email.toLowerCase().includes(s) ||
        vc.customer.phone.toLowerCase().includes(s)
      );
    }
    return { data: filtered };
  },

  // Add or link customer
  async addOrLinkCustomer(vendor_id: string, req: AddOrLinkCustomerRequest) {
    // 1. Find by email or phone
    let { data: customer } = await supabase.from('customers').select('*').or(`email.eq.${req.email},phone.eq.${req.phone}`).single();
    if (!customer) {
      // Create global customer
      const { data: newCustomer, error: createError } = await supabase.from('customers').insert({
        full_name: req.full_name,
        email: req.email,
        phone: req.phone
      }).select().single();
      if (createError) return { error: createError };
      customer = newCustomer;
    }
    // 2. Link to vendor (if not already linked)
    let { data: vendorCustomer } = await supabase.from('vendor_customers').select('*').eq('vendor_id', vendor_id).eq('customer_id', customer.id).single();
    if (!vendorCustomer) {
      const { data: newVC, error: linkError } = await supabase.from('vendor_customers').insert({
        vendor_id,
        customer_id: customer.id,
        tags: req.tags || [],
        display_name: req.full_name
      }).select().single();
      if (linkError) return { error: linkError };
      vendorCustomer = newVC;
    } else if (req.tags) {
      // Update tags if provided
      await supabase.from('vendor_customers').update({ tags: req.tags }).eq('id', vendorCustomer.id);
    }
    return { data: { customer, vendor_customer: vendorCustomer } };
  },

  // Get customer profile (global, vendor, measurements)
  async getCustomerProfile(vendor_id: string, customer_id: string) {
    const { data: customer, error: customerError } = await supabase.from('customers').select('*').eq('id', customer_id).single();
    if (customerError) return { error: customerError };
    const { data: vendorCustomer, error: vcError } = await supabase.from('vendor_customers').select('*').eq('vendor_id', vendor_id).eq('customer_id', customer_id).single();
    if (vcError) return { error: vcError };
    const { data: measurements } = await supabase.from('measurements').select('*').eq('vendor_id', vendor_id).eq('customer_id', customer_id).order('created_at', { ascending: false }).limit(1).single();
    return { data: { customer, vendor_customer: vendorCustomer, measurements } };
  },

  // Update vendor-specific customer info
  async updateVendorCustomer(vendor_id: string, customer_id: string, req: UpdateCustomerRequest) {
    const update: any = {};
    if (req.full_name) update.display_name = req.full_name;
    if (req.tags) update.tags = req.tags;
    const { data, error } = await supabase.from('vendor_customers').update(update).eq('vendor_id', vendor_id).eq('customer_id', customer_id).select().single();
    return { data, error };
  },

  // Add or update measurements
  async upsertMeasurements(vendor_id: string, customer_id: string, req: MeasurementRequest) {
    // Overwrite existing (upsert by vendor_id + customer_id)
    const { data, error } = await supabase.from('measurements').upsert({
      vendor_id,
      customer_id,
      ...req
    }, { onConflict: 'vendor_id,customer_id' }).select().single();
    return { data, error };
  },

  // Unlink customer from vendor
  async unlinkCustomer(vendor_id: string, customer_id: string) {
    const { error } = await supabase.from('vendor_customers').delete().eq('vendor_id', vendor_id).eq('customer_id', customer_id);
    return { success: !error, error };
  }
}; 